/*
 * @(#)AMControlFlowFigure.java  1.0  2006-12-19
 *
 * Copyright (c) 2006 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm.am.diagram;

import ch.hslu.cm.am.model.*;
import ch.hslu.cm.simulation.*;
import ch.hslu.cm.*;
import java.beans.*;
import java.awt.event.*;
import java.io.*;
import java.util.*;
import javax.swing.undo.*;
import javax.swing.*;
import java.awt.geom.*;
import static org.jhotdraw.draw.AttributeKeys.*;
import org.jhotdraw.draw.*;
import org.jhotdraw.draw.decoration.ArrowTip;
import org.jhotdraw.draw.decoration.GeneralPathTip;
import org.jhotdraw.draw.decoration.LineDecoration;
import org.jhotdraw.draw.event.FigureAdapter;
import org.jhotdraw.draw.event.FigureEvent;
import org.jhotdraw.draw.layouter.LocatorLayouter;
import org.jhotdraw.draw.locator.BezierLabelLocator;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;

/**
 * Represents the FIGURE (graphical, visual representation) of an ControlFlow
 * element in an Activity Diagram.
 * <p>
 * A control flow is an edge that starts an activity node after the previous
 * one is finished. Objects and data cannot pass along a control flow edge.
 * <p>
 * A control flow is notated by an arrowed line connecting two nodes.
 * <p>
 * For further information, consult the official UML
 * <a href="http://www.omg.org/docs/formal/05-07-04.pdf">online resource</a>
 * of the Object Management Group (OMG):<br>
 * Unified Modeling Language: Superstructure, version 2.0, formal/05-07-04,
 * page 344 et sqq.
 *
 * @author Florian Padrun
 * @version 1.0 2006-12-19 Created.
 */
public class AMControlFlowFigure extends LabeledLineConnectionFigure
        implements DiagramFigure, PropertyChangeListener {
    /** Figure model. */
    private AMControlFlow model;
    
    /** Specifies the dimension of the arrowTip of an control flow element */
    private static ArrowTip arrowTip = new ArrowTip(15d / 180d * Math.PI, 10,
            10, true, true, true);
    
    /** Specifies the decoration of a line. */
    private static LineDecoration connectedWithNodeTip;
    static
    {
        Path2D.Double p = new Path2D.Double();
        p.moveTo(-2, 0);
        p.lineTo(2, 0);
        p.lineTo(2,  12);
        p.lineTo(-2, 12);
        p.lineTo(-2, 0);
        connectedWithNodeTip = new GeneralPathTip(p, 0, true, false, true);
    }
    
    /**
     * This adapter is used, to connect a TextFigure for an attribute with the
     * AMControlFlow model.
     */
    private class LabelAdapter extends FigureAdapter {
        private boolean owner;
        
        public LabelAdapter(boolean owner) {
            this.owner = owner;
        }
        
    @Override
        public void attributeChanged(final FigureEvent e) {
            if (e.getAttribute().equals(AttributeKeys.TEXT)) {
                String newValue = (String) e.getNewValue();
                model.setGuardCondition(newValue);
                
                fireUndoableEditHappened(new AbstractUndoableEdit() {
                    @Override
                    public String getPresentationName() {
                        ResourceBundleUtil labels
                                = ResourceBundleUtil.getBundle(
                                "ch.hslu.cm.am.Labels", Locale.getDefault());
                        return labels.getString("guardCondition");
                    }
                    
                    @Override
                    public void undo() throws CannotUndoException {
                        super.undo();
                        model.setGuardCondition((String) e.getOldValue());
                    }
                    
                    @Override
                    public void redo() throws CannotUndoException {
                        super.redo();
                        model.setGuardCondition((String) e.getNewValue());
                    }
                });
            }
        }
    }
    
    /** Creates a new instance. */
    public AMControlFlowFigure() {
        setLayouter(new LocatorLayouter());
        setModel(new AMControlFlow());
        STROKE_COLOR.set(this, ActivityDiagram.STROKE_COLOR);
        STROKE_WIDTH.set(this, ActivityDiagram.STROKE_WIDTH);
        START_DECORATION.set(this, null);
        END_DECORATION.set(this, arrowTip);
        FONT_BOLD.set(this, true);
        
        setAttributeEnabled(AttributeKeys.END_DECORATION, false);
        setAttributeEnabled(AttributeKeys.START_DECORATION, false);
        setAttributeEnabled(AttributeKeys.STROKE_DASHES, false);
        setAttributeEnabled(AttributeKeys.STROKE_TYPE, false);
        setAttributeEnabled(AttributeKeys.STROKE_WIDTH, false);
        setAttributeEnabled(AttributeKeys.FONT_BOLD, false);
        setAttributeEnabled(AttributeKeys.FONT_ITALIC, false);
        setAttributeEnabled(AttributeKeys.FONT_UNDERLINE, false);
        setAttributeEnabled(TEXT_COLOR, false);
    }
    
    /**
     * Gets the model of the figure.
     * @return model
     */
    @Override
    public AMControlFlow getModel() {
        return model;
    }
    
    /**
     * Sets the model of the figure.
     * @param m AMControlFlow
     */
    public void setModel(AMControlFlow m) {
        if (model != null) {
            model.removePropertyChangeListener(this);
        }
        model = m;
        
        if (model != null) {
            model.addPropertyChangeListener(this);
        }
        updateLabels();
    }
    
    /**
     * Informs the figure, that it has been added to the specified drawing.
     * The figure must inform all FigureListeners that it has been added.
     * @param drawing Drawing
     */
    @Override
    public void addNotify(Drawing drawing) {
        super.addNotify(drawing);
        if ((drawing instanceof Diagram) && getModel() != null) {
            getSimulation().add(getModel());
        }
    }
    
    /**
     * Informs the figure, that it has been removed from the specified drawing.
     * The figure must inform all FigureListeners that it has been removed.
     * @param drawing Drawing
     */
    @Override
    public void removeNotify(Drawing drawing) {
        if (getDrawing() != null && getModel() != null) {
            getSimulation().remove(getModel());
        }
        super.removeNotify(drawing);
    }
    
    /**
     * Updates the labels of the figure.
     */
    private void updateLabels() {
        ResourceBundleUtil labels = ActivityModel.labels;
        
        willChange();
        basicRemoveAllChildren();
        
        if (model.isGuarded()) {
            TextFigure guardCondition
                    = new TextFigure(model.getGuardCondition());
            guardCondition.set(LocatorLayouter.LAYOUT_LOCATOR,
                    new BezierLabelLocator(0.5, -90d / 180d * Math.PI, 2));
            guardCondition.addFigureListener(new LabelAdapter(true));
            guardCondition.set(FONT_BOLD, true);
            basicAdd(guardCondition);
        }
        
        for (Figure f : getChildren()) {
            AbstractAttributedFigure label = (AbstractAttributedFigure) f;
            label.setAttributeEnabled(STROKE_COLOR, false);
            label.setAttributeEnabled(STROKE_WIDTH, false);
            label.setAttributeEnabled(STROKE_DASHES, false);
            label.setAttributeEnabled(STROKE_MITER_LIMIT, false);
            label.setAttributeEnabled(STROKE_JOIN, false);
            label.setAttributeEnabled(FONT_BOLD, false);
            label.set(TEXT_COLOR, get(TEXT_COLOR));
            label.set(FONT_FACE, get(FONT_FACE));
            label.set(FONT_ITALIC, get(FONT_ITALIC));
            label.set(FONT_UNDERLINE,
                    get(FONT_UNDERLINE));
            label.set(FONT_SIZE, get(FONT_SIZE));
        }
        layout();
        changed();
    }
    
    /**
     * Returns a collection of Action's for the specified location.
     * @param p Point2D.Double
     */
    @Override
    public Collection<Action> getActions(Point2D.Double p) {
        final ResourceBundleUtil labels
                = ResourceBundleUtil.getBundle("ch.hslu.cm.am.Labels",
                Locale.getDefault());
        
        LinkedList<Action> actions = new LinkedList<Action>();
        Action action;
        
        Figure item;
        
        action = new AbstractAction(model.isGuarded()
        ? labels.getString("hideGuardCondition")
        : labels.getString("showGuardCondition")) {
            @Override
            public void actionPerformed(ActionEvent event) {
                model.setGuarded(! model.isGuarded());
            }
        };
        actions.add(action);
        
        return actions;
    }
    
    /**
     * This method gets called when a bound property is changed.
     * @param evt PropertyChangeEvent
     */
    @Override
    public void propertyChange(PropertyChangeEvent evt) {
        updateLabels();
    }
    
    /**
     * Gets the container for diagram figures.
     * @return diagram
     */
    private Diagram getDiagram() {
        return (Diagram) getDrawing();
    }
    
    /**
     * Gets the simulation (collection of elements) of the diagram.
     * @return simulation
     */
    private Simulation getSimulation() {
        return (getDiagram() == null) ? null : getDiagram().getSimulation();
    }
    
    /**
     * Creates and returns a copy of the figure object.
     * @return that
     */
    @Override
    public AMControlFlowFigure clone() {
        AMControlFlowFigure that = (AMControlFlowFigure) super.clone();
        that.setModel(this.getModel().clone());
        return that;
    }
    
    /**
     * Handles the disconnection of a connection.
     * @param start Figure
     * @param end Figure
     */
    protected void handleDisconnect(Figure start, Figure end) {
        model.setEnd(null);
        model.setStart(null);
        updateLabels();
    }
    
    /**
     * Handles the connection of a connection.
     * @param start Figure
     * @param end Figure
     */
    protected void handleConnect(Figure start, Figure end) {
        if ((start instanceof DiagramFigure) && (end instanceof DiagramFigure)) {
            DiagramFigure sf = (DiagramFigure) start;
            DiagramFigure ef = (DiagramFigure) end;
            
            model.setEnd((SimulatedElement) ef.getModel());
            model.setStart((SimulatedElement) sf.getModel());
            
            updateLabels();
        }
    }
    
    /**
     * Checks if two figures can be connected.
     * @param start Figure
     * @param end Figure
     */
    public boolean canConnect(Figure start, Figure end) {
        if ((start instanceof DiagramFigure) && end instanceof DiagramFigure) {
            DiagramFigure startdf = (DiagramFigure) start;
            DiagramFigure enddf = (DiagramFigure) end;
            if ((startdf.getModel() instanceof SimulatedElement)
            && (enddf.getModel() instanceof SimulatedElement)) {
                return model.canConnect((SimulatedElement) startdf.getModel(),
                        (SimulatedElement) enddf.getModel());
            }
        }
        return false;
    }
    
    /**
     * Checks if a figure can be connected.
     * @param start Figure
     */
    public boolean canConnect(Figure start) {
        if (start instanceof DiagramFigure) {
            DiagramFigure startdf = (DiagramFigure) start;
            if (startdf.getModel() instanceof SimulatedElement) {
                return model.canConnect((SimulatedElement) startdf.getModel());
            }
        }
        return false;
    }
    
    /**
     * Reads the persistent figure object.
     * @param in DOMInput
     */
    @Override
    public void read(DOMInput in) throws IOException {
        in.openElement((in.getElementCount("model") == 1) ? "model" : "Model");
        setModel((AMControlFlow) in.readObject(0));
        in.closeElement();
        readPoints(in);
        readAttributes(in);
    }
    
    /**
     * Writes the figure object in a persisten state.
     * @param out DOMOutput
     */
    @Override
    public void write(DOMOutput out) throws IOException {
        out.openElement("Model");
        out.writeObject(getModel());
        out.closeElement();
        writePoints(out);
        writeAttributes(out);
    }
    
    /**
     * Gets the layer of the figure in the diagram.
     * @return CONNECTION_LAYER
     */
    @Override
    public int getLayer() {
        return ActivityDiagram.CONNECTION_LAYER;
    }
}